import os
import ssl
import json
import time
import asyncio
import threading
import websockets
import sys
import subprocess
from http.server import HTTPServer, SimpleHTTPRequestHandler
from gpiozero import AngularServo, LED
from gpiozero.pins.pigpio import PiGPIOFactory

# --- 1. グローバル変数の宣言と初期化 ---
# 変数名だけ用意
factory = None
servo_ud = None
servo_rl = None
ssr = None
httpd = None
running = True
is_homing = False  
light_ = 0
rl_ang, rl_dir = 0, 0
ud_ang, ud_dir = 0, 0

# --- 2. HTTPハンドラ（ブラウザに画面を出す役目） ---
class RestrictedHTTPHandler(SimpleHTTPRequestHandler):
    def do_GET(self):
        allowed_extensions = {'.html', '.css', '.js', '.ico'}
        _, ext = os.path.splitext(self.path)
        if self.path == "/" or ext.lower() in allowed_extensions:
            return super().do_GET()
        else:
            self.send_error(403, "Access Denied.")

# --- 3. モーター制御ロジック（別スレッドで動く） ---
def motor_loop(servo, limit):
    global ud_ang, rl_ang, ud_dir, rl_dir, running, is_homing
    step = 0.5
    is_ud = (servo == servo_ud)
    
    servo.detach() # 最初は脱力

    while running:
        cur_dir = ud_dir if is_ud else rl_dir
        if cur_dir != 0:
            angle = ud_ang if is_ud else rl_ang
            angle += (cur_dir * step)
            angle = max(-limit, min(limit, angle))
            servo.angle = angle
            if is_ud: ud_ang = angle
            else: rl_ang = angle
            time.sleep(0.05)
            continue 
        else:
            if not is_homing and servo.value is not None:
                servo.detach()
            time.sleep(0.1)

# --- 4. WebSocket ハンドラー（スマホの命令をさばく） ---
async def ws_handler(websocket, path=None):
    global light_, rl_ang, rl_dir, ud_ang, ud_dir, running, is_homing
    try:
        async for message in websocket:
            data = json.loads(message)
            cmd = data.get("cmd")
            if cmd == 'get_pos':
                pos_str = f"{int(-ud_ang)}/{int(-rl_ang)}"
                await websocket.send(json.dumps(pos_str))
            elif cmd in ['up', 'down', 'left', 'right', 'stop']:
                if cmd == 'up':      ud_dir = -1
                elif cmd == 'down':    ud_dir = 1
                elif cmd == 'right':   rl_dir = -1
                elif cmd == 'left':    rl_dir = 1
                elif cmd == 'stop':    ud_dir = rl_dir = 0
            elif cmd == 'light':
                light_ ^= 1
                ssr.on() if light_ else ssr.off()
            elif cmd == 'home':
                is_homing = True 
                rl_ang = ud_ang = rl_dir = ud_dir = 0
                servo_ud.angle = 0
                servo_rl.angle = 0
                time.sleep(1.5) 
                is_homing = False 
                pos_str = "0/0"
                await websocket.send(json.dumps(pos_str))                
    except Exception:
        pass

# --- 5. メイン実行部（ここが「本番」の順番） ---
if __name__ == '__main__':
    # ① まずハードウェアを準備する
    factory = PiGPIOFactory()
    servo_ud = AngularServo(21, min_angle=-90, max_angle=90, min_pulse_width=0.0005, max_pulse_width=0.0024, pin_factory=factory, initial_angle=0)
    servo_rl = AngularServo(20, min_angle=-90, max_angle=90, min_pulse_width=0.0005, max_pulse_width=0.0024, pin_factory=factory, initial_angle=0)
    ssr = LED(26) 
    
    time.sleep(1.5) # 最初の位置合わせ完了を待つ

    # ② 次に「モータースレッド」を裏側で走らせる
    # これをサーバー起動（③）の「前」に書くのが重要！
    threading.Thread(target=motor_loop, args=(servo_ud, 60), daemon=True).start()
    threading.Thread(target=motor_loop, args=(servo_rl, 70), daemon=True).start()
    
    # ③ サーバーの設定と起動
    loop = asyncio.new_event_loop()
    asyncio.set_event_loop(loop)
    ssl_ctx = ssl.SSLContext(ssl.PROTOCOL_TLS_SERVER)
    base_path = os.path.dirname(os.path.abspath(__file__))
    ssl_ctx.load_cert_chain(os.path.join(base_path, 'localhost.pem'), os.path.join(base_path, 'localhost-key.pem'))

    async def start_servers():
        global httpd
        async with websockets.serve(ws_handler, "0.0.0.0", 8700, ssl=ssl_ctx):
            # HTTPSサーバーをさらに別スレッドで走らせる
            httpd = HTTPServer(('', 443), RestrictedHTTPHandler)
            httpd.socket = ssl_ctx.wrap_socket(httpd.socket, server_side=True)
            threading.Thread(target=httpd.serve_forever, daemon=True).start()
            
            print("--- System Ready ---")
            while running: await asyncio.sleep(1)

    # 実行
    try:
        loop.run_until_complete(start_servers())
    except KeyboardInterrupt:
        print("\nStopping...")
        running = False
        if httpd:
            stop_thread = threading.Thread(target=httpd.shutdown)
            stop_thread.start()
            stop_thread.join()
    finally:
        if httpd: httpd.server_close()
        servo_ud.close()
        servo_rl.close()
        ssr.off()
        sys.exit(0)